/*******************************************************************************
 * Copyright (C) 2009-2020 Human Media Interaction, University of Twente, the Netherlands
 *
 * This file is part of the Articulated Social Agents Platform BML realizer (ASAPRealizer).
 *
 * ASAPRealizer is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License (LGPL) as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * ASAPRealizer is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with ASAPRealizer.  If not, see http://www.gnu.org/licenses/.
 ******************************************************************************/
package hmi.mixedanimationenvironment;

import hmi.animation.VJoint;
import hmi.animation.VObjectTransformCopier;
import hmi.util.AnimationSync;
import hmi.util.PhysicsSync;

import java.util.ArrayList;

import net.jcip.annotations.GuardedBy;

/**
 * Manages the playback and physical simulation of multiple AnimationPlayers
 * 
 * slightly modified for the new environment classes.
 * @author Herwin van Welbergen
 * @author Dennis Reidsma
 * 
 */
public class MixedAnimationPlayerManager
{
    @GuardedBy("this")
    private ArrayList<VObjectTransformCopier> votcs = new ArrayList<VObjectTransformCopier>();

    @GuardedBy("this")
    private ArrayList<MixedAnimationPlayer> players = new ArrayList<MixedAnimationPlayer>();

    @GuardedBy("this")
    private double prevTime = 0;

    @GuardedBy("PhysicsSync.getSync(),AnimationSync.getSync()")
    // TODO: both needed??
    private PhysicsCallback physicsCallback;

    private final float physicsStepTime;
    public static final float DEFAULT_PHYSICS_STEPTIME = 0.003f;
    private ArrayList<VJoint> animationRootJoints = new ArrayList<VJoint>();
    private ArrayList<VJoint> currentRootJoints = new ArrayList<VJoint>();

    /**
     * 
     * In each frame:
     * 1. Run all animation players, in each animation player: <br>
     * a Copy next to cur, cur to prev <br>
     * b Run procedural animations on next <br>
     * c Calculate the forces and torques generated by the proc motion <br>
     * 2. Do a physics step <br>
     * 3. Copy physical body rotations to cur and next <br>
     * 4. Copy curr to the animation tree <br>
     * 
     */
    public MixedAnimationPlayerManager(PhysicsCallback pb)
    {
        this(pb, DEFAULT_PHYSICS_STEPTIME);
    }

    public MixedAnimationPlayerManager(PhysicsCallback pb, float physicsStepTime)
    {
        this.physicsStepTime = physicsStepTime;
        physicsCallback = pb;
    }

    public void addRootJoints(VJoint currRoot, VJoint animationRoot)
    {
        VObjectTransformCopier votc = VObjectTransformCopier.newInstanceFromMatchingVObjectLists(currRoot.getParts(),
                animationRoot.getParts(), "TR");
        votcs.add(votc);
        /*
         * VOTC operates on sid equals OR id equals OR name equals. This gives
         * problems if there are three humanoids in the tree -- everything will
         * be copied to the first humanoid. therefore we use the ugly hacked
         * version of VOTC constructor. ASSUMPTION: the humanoidroots are added
         * directly under animationRoot
         */
        animationRootJoints.add(animationRoot);
        currentRootJoints.add(currRoot);
    }

    /**
     * Removes the animationplayer. Make sure the VJoint controlled by p is removed from the animationRootJoint.
     */
    public synchronized void removeAnimationPlayer(MixedAnimationPlayer p, VJoint currRoot, VJoint animationRoot)
    {
        players.remove(p);
        int i = animationRootJoints.indexOf(animationRoot);
        animationRootJoints.remove(i);
        currentRootJoints.remove(i);
        votcs.remove(i);
    }

    public synchronized void addAnimationPlayer(MixedAnimationPlayer p, VJoint currRoot, VJoint animationRoot)
    {
        players.add(p);
        addRootJoints(currRoot, animationRoot);
    }

    public synchronized void time(double currentTime)
    {
        double time = (currentTime - prevTime);

        // handle resetSimulation
        if (time < 0)
        {
            prevTime = currentTime;
            return;
        }

        if (time > physicsStepTime)
        {
            while (time > physicsStepTime)
            {
                prevTime += physicsStepTime;
                for (MixedAnimationPlayer p : players)
                {
                    p.playStep(prevTime);
                }                
                synchronized (PhysicsSync.getSync())
                {
                    physicsCallback.time(physicsStepTime);
                }
                time -= physicsStepTime;                
            }
        }
        else
        {
            return;
        }

        for (MixedAnimationPlayer p : players)
        {
            p.copyPhysics();
        }

        synchronized (AnimationSync.getSync())
        {
            for (VObjectTransformCopier votc : votcs)
            {
                votc.copyConfig();
            }
        }
    }

    /**
     * @return the PHYSICS_STEPTIME
     */
    public float getH()
    {
        return physicsStepTime;
    }

    /**
     * @return the prevTime
     */
    public synchronized double getPrevTime()
    {
        return prevTime;
    }
}
